#!/usr/bin/perl -w
use strict;

#
# Convert Clark Kimberling's Encyclopedia of Triangle Centers (ETC)
# into java functions of the form:
#   public static double[] calcTrilinears(int i, double a, double b, double c)
#   public static double[] calcBarycentrics(int i, double a, double b, double c)
# where a,b,c are the triangle side lengths.
#

my $verbose = 0; # debugging level


sub javafy($$)
{
    my ($s,$i) = @_;    # $i is just for messages XXX I think line number would be more helpful, or maybe both

    $s =~ s/$/ /g; # add a trailing space temporarily, so we can easily parse groups of words

    # XXX fix typos which should be reported to Clark Kimberling
    # XXX also: in glossary, definition of isogonal conjugate, L(B) should be L(C)
    {
        $s =~ s/B:/B :/g # typo in X(20) and X(35) X(36)
            and print STDERR "typo in $i: $s\n";
        $s =~ s/\bcsc *$/csc C /g # typo in X(43)
            and print STDERR "typo in $i: $s\n";
        $s =~ s/(\[1 [+-] cos\(C - A\)) (:)/$1] $2/g # typo in X(59) and X(60) (twice each)
            and print STDERR "typo in $i: $s\n";
        $s =~ s/(: cos B sin<sup>2<\/sup>\(C - A\))\]/$1/g # typo in X(125)
            and print STDERR "typo in $i: $s\n";
        $s =~ s/(: sec C\/2) *(f)/$1 = $2/g # typo in X(174)
            and print STDERR "typo in $i: $s\n";
        $s =~ s/(\(a - b) *$/$1) /g # typo in X(190)
            and print STDERR "typo in $i: $s\n";
        $s =~ s/\. $//g # typo in X(191), X(481) and X(482)
            and print STDERR "typo in $i: $s\n";
        $s =~ s/C = &omega;/C + &omega;/g # typo in X(232) (twice)
            and print STDERR "typo in $i: $s\n";
        $s =~ s/(cos\(B \+ &omega;\))(<sup>2<\/sup>)/$1 : c$2/g # typo in X(237)
            and print STDERR "typo in $i: $s\n";
        $s =~ s/: csc C\)/: csc C/g # typo in X(311)
            and print STDERR "typo in $i: $s\n";
        $s =~ s/(: \(csc B cot B\))\)/$1/g # typo in X(337)
            and print STDERR "typo in $i: $s\n";
        # XXX there's more typos near the citation in the alt versions of 481 and 482
        $s =~ s/(bf\(b,c,a\)) (cf\(c,a,b\))/$1 : $2/g # typo in X(515) and X(518)
            and print STDERR "typo in $i: $s\n";
        $s =~ s/(b\/f\(b,c,a\)) (c\/f\(c,a,b\))/$1 : $2/g # typo in X(516)
            and print STDERR "typo in $i: $s\n";
        $s =~ s/: \//: /g # typo in X(553)
            and print STDERR "typo in $i: $s\n";
        $s =~ s/sin C\(/sin C (/g # typo in X(650) and X(652)
            and print STDERR "typo in $i: $s\n";
        $s =~ s/(\(c - a\)) (c<sup>2)/$1 : $2/g # typo in X(692)
            and print STDERR "typo in $i: $s\n";
        $s =~ s/(\(c - a\)) (c<sup>3)/$1 : $2/g # typo in X(692)
            and print STDERR "typo in $i: $s\n";
        $s =~ s/S<sub>B<\/sub>S<sub>C<\/sub>\(S<sub>B<\/sub> - S<sub>C<\/sub> : S<sub>C<\/sub>S<sub>A<\/sub>\(S<sub>C<\/sub> - S<sub>A<\/sub> : S<sub>A<\/sub>S<sub>B<\/sub>\(S<sub>A<\/sub> - S<sub>B<\/sub>/S<sub>B<\/sub>S<sub>C<\/sub>(S<sub>B<\/sub> - S<sub>C<\/sub>) : S<sub>C<\/sub>S<sub>A<\/sub>(S<sub>C<\/sub> - S<sub>A<\/sub>) : S<sub>A<\/sub>S<sub>B<\/sub>(S<sub>A<\/sub> - S<sub>B<\/sub>)/g # typo in X(2501)
            and print STDERR "typo in $i: $s\n";
    }

    # get rid of (cf., X(39)) and such
    $s =~ s/ *\(cf\.,? X\(\d+\)\) */ /;

    # get rid of Peter J.C. Moses and Joe Goggins and Darij Grinberg
    $s =~ s/ *\(Peter J. C. Moses, [^)]+\) */ /;
    $s =~ s/ *\(Joe Goggins, [^)]+\) */ /;
    $s =~ s/ *\(Darij Grinberg, [^)]+\) */ /;

    # If it has the definition of W that we use, remove it
    $s =~ s/, where W = \(a<sup>2<\/sup> \+ b<sup>2<\/sup> \+ c<sup>2<\/sup>\)\/\(4\*area\) *$/ /g;

    # Substitute the definition of A' in X(848)...
    # XXX this needs further tweaking, g needs 6 params I guess
    $s =~ s/, (where )?A' = 2&pi;a\/\(a \+ b \+ c\) *$/ /g;
    $s =~ s/A'/(2*Math.PI*a\/(a+b+c)/g;

    # spell out primes and double primes
    $s =~ s/a'/a_prime/g;
    $s =~ s/a"/a_prime_prime/g;

    $s =~ s/\[/(/g; # change square brackets to parens
    $s =~ s/\]/)/g; # change square brackets to parens

    $s =~ s/[Aa]rea(\(ABC\))/area /g;

    # expand S<sub>A</sub> to S_A, etc.
    $s =~ s/\bS<sub>A<\/sub>/S_A /g;
    $s =~ s/\bS<sub>B<\/sub>/S_B /g;
    $s =~ s/\bS<sub>C<\/sub>/S_C /g;


    # change positive powers of trig functions to my special functions
    $s =~ s/\b(sin|cos|tan|cot|sec|csc)<sup>([0-9])<\/sup>/$1$2 /g;
    # change all other powers to the pow function, with a space beforehand just in case
    $s =~ s/(\([^()]*\([^()]*\)[^()]*\))<sup> *(-?) *([0-9])<\/sup>/ pow($1,$2$3)/g;
    $s =~ s/(\([^()]*\([^()]*\)[^()]*\))<sup> *(-?) *([0-9]\/[0-9])<\/sup>/ pow($1,$2$3.)/g; # XXX note appending . to avoid integer division
    $s =~ s/(\([^()]*\)) *<sup> *(-?) *([0-9])<\/sup>/ pow($1,$2$3)/g;
    $s =~ s/([a-zA-Z]) *<sup> *2 *<\/sup>/($1*$1)/g;
    $s =~ s/([a-zA-Z]) *<sup> *3 *<\/sup>/($1*$1*$1)/g;
    $s =~ s/([a-zA-Z]) *<sup> *4 *<\/sup>/($1*$1*$1*$1)/g;
    $s =~ s/([a-zA-Z]) *<sup> *5 *<\/sup>/($1*$1*$1*$1*$1)/g;
    $s =~ s/([a-zA-Z]) *<sup> *6 *<\/sup>/($1*$1*$1*$1*$1*$1)/g;
    $s =~ s/([a-zA-Z]) *<sup> *7 *<\/sup>/($1*$1*$1*$1*$1*$1*$1)/g;
    $s =~ s/([a-zA-Z]) *<sup> *8 *<\/sup>/($1*$1*$1*$1*$1*$1*$1*$1)/g;
    $s =~ s/([a-zA-Z]) *<sup> *(-?) *([0-9])<\/sup>/ pow($1,$2$3)/g;
    $s =~ s/([a-zA-Z]) *<sup> *(-?) *([0-9]\/[0-9])<\/sup>/ pow($1,$2$3.)/g; # note appending . to avoid integer division
    # XXX maybe should do concat-to-* first so can do the above only at word boundaries?

    # Put parens around trig args, and remove any space between fun and paren.
    # Recall (?=pattern) is the zero-width positive lookahead assertion.
    $s =~ s/\b((sin|cos|tan|cot|sec|csc)[0-9]?)\b *\(([^(),]+)\)( |(?=\)))/$1($3) /g; # \b means word boundary
    $s =~ s/\b((sin|cos|tan|cot|sec|csc)[0-9]?)\b ([^(), ]+)( |(?=\)))/$1($3) /g; # \b means word boundary

    # Change concatenation to *...
    $s =~ s/\) *\(/) * (/g;
    $s =~ s/\) *([A-Za-z])/) * $1/g;

    $s =~ s/\b([abcuvwxyzABC1-5])([abcfguvwxyzABC]+)\b/$1*$2/g;
    $s =~ s/\b([abcuvwxyzABC1-5])([abcfguvwxyzABC]+)\b/$1*$2/g;
    $s =~ s/\b([abcuvwxyzABC1-5])([abcfguvwxyzABC]+)\b/$1*$2/g;

    $s =~ s/\b([abcuvwxyz1-5]|S_[ABC])\b *([a-zA-Z(])/$1*$2/g;
    $s =~ s/\b([abcuvwxyz1-5]|S_[ABC])\b *([a-zA-Z(])/$1*$2/g; # again, to catch consecutive

    # Put Math. before names functions that are in the math library (but not the ones with numbers appended)
    $s =~ s/\b(pow|sin|cos|tan)\b/Math.$1/g; # \b means word boundary

    # change &pi; to Math.PI
    $s =~ s/&pi;/Math.PI/g;
    $s =~ s/2 ?Math.PI/2*Math.PI/g;

    # Brocard angle...
    $s =~ s/&omega;/omega/g;
    $s =~ s/2 ?omega/2*omega/g;

    # Change this type of comma to a "where", and protect the colons...
    $s =~ s/, x : y : z = X/ where x COLON y COLON z = X/g;

    # change colons to commas
    $s =~ s/:/,/g;

    # Get the colons back...
    $s =~ s/ COLON /:/g;

    $s =~ s/ +$//g; # delete the trailing space we added
    return $s;
} # javafy


    my @ungotlines = ();
sub getline()
{
    if (@ungotlines > 0)
    {
        ++$.; # adjust input line number for error messages
        return pop(@ungotlines);
    }
    else
    {
        return <>;
    }
}
sub ungetline($)
{
    my ($line) = @_;
    push(@ungotlines, $line);
    --$.; # adjust input line number for error messages
}

{
    my @names = ();
    my @trilinears = ();
    my @barycentrics = ();
    my @auxfunctions = ();

    {
        my @noBarycentricsOkay = (1166..1216, 1218..1275, 1286..1311, 1342..1343, 1512..1568, 1601..1651, 1670..1671);
        my $noBarycentrics = 0; # flag

        my $i = undef;
        while (my $line = getline())
        {
            $line =~ s/[\r\n]//g;

            while ($line =~ /^<h3\b.*<BR>$/)
            {
                $line =~ /X(192)/ or die; # intentional inconsistency at 192, maybe should make legit
print(STDERR "XXXXXX weirdness at 192; make sure it comes out right\n");
                my $nextline = getline();
                $line .= $nextline;
                $line =~ s/[\r\n]//g;
            }

            while ($line =~ /^Trilinears/
                && $line !~ /^Trilinears found /) # XXX fooey, X(2671)
            {
                my $nextline = getline();
                if ($nextline =~ /^Barycentrics/)
                {
                    ungetline($nextline);
                    last;
                }
                elsif ($nextline =~ /^X\(\d+\) lies on /
                    || $nextline =~ /^X\(\d+\) is the /
                    || $nextline =~ /^X\(\d+\) = /
                    ) # XXX maybe just /^X\(\d+\) /?
                    # XXX note typo "thise line" in X(1540)
                {
                    print(STDERR "Looks like $i doesn't have a barycentrics!?\n");
                    defined($i) && grep {$_==$i} @noBarycentricsOkay or die;
                    $noBarycentrics = 1; #set flag
                    ungetline($nextline);
                    last;
                }
                else
                {
                    $nextline =~ s/^Trilinears/=/;
                    $line .= $nextline;
                    $line =~ s/[\r\n]//g;
                }
            }
            while ($line =~ /^Barycentrics/) # just for X(144)... relies on it not having the following thing (where or , or :); in fact it relies on being just one line
            {
                my $nextline = getline();
                my $nextnextline = getline();
                if ($nextline =~ /^<P>\s*$/
                 && $nextnextline =~ /^Barycentrics/)
                {
                    $i == 144 or
                    $i == 1316 or die; # typo/inconsistency at X(144), X(1316)
print(STDERR "XXXXXX inconsistency at X($i); make sure it comes out right\n");
                    $nextnextline =~ s/^Barycentrics/=/;
                    $line .= $nextline . $nextnextline;
                    $line =~ s/[\r\n]//g;
                }
                else
                {
                    ungetline($nextnextline);
                    ungetline($nextline);
                    last;
                }
            }
            while ($line =~ /^Barycentrics.*( where|,|:) ?(<BR> ?)?$/)
            {
                my $nextline = getline();
                $line .= $nextline;
                $line =~ s/[\r\n]//g;
            }


            $line =~ s/(&nbsp;?|<BR>|<P>|\s)+/ /g; # the ? is for typo at X(74) where the ; is missing
            $line =~ s/ id=97"X"/ id="X97"/; # obvious typo at X(97) XXX report it

            if ($line =~ /^<h3>X\((\d+)\).* NAME="(.*)"/)
            {
                !defined($i) or die;
                $i = $1;
                !defined($names[$i]) or die;
                $names[$i] = $2;
            }
            # in version I downloaded Wed Oct  6 17:53:40 PDT 2004,
            # the ids and anchors are new.
            elsif ($line =~ /^<h3( id="X\d+")?>X\((\d+)\) ?=? ?([^<]*)(<A NAME=".*"><\/A>)?<\/h3>/
                || $line =~ /^<h3( id="X\d+")?>X\((\d+)\)< ?=? ?([^<]*)(<A NAME=".*"><\/A>)?<\/h3>/) # XXX typo, extra < for X(917)
            {
                !defined($i) or die;
                $i = $2;
                !defined($names[$i]) or die;
                $names[$i] = $3;
                if ($names[$i] eq "")
                {
                    #print STDERR "warning: $number has no name\n";
                    $names[$i] = "X($i)";
                }
            }
            elsif ($line =~ /^Trilinears (.*)\s*$/
                && $line !~ /^Trilinears found /) # XXX fooey, X(2671)
            {
                defined($i) && !defined($trilinears[$i]) or die;
                $trilinears[$i] = [javafy($1,$i)];
                if ($noBarycentrics)
                {
                    $noBarycentrics = 0;
                    $i = undef;
                }
            }
            elsif ($line =~ /^(f\(a,b,c\) : f\(b,c,a\) : f\(c,a,b\), where f\(a,b,c\) = \(a<sup>2<\/sup> \+ 4\*area\(ABC\)\))\/a $/)
            {
                # XXX typo on X(590)...
                defined($i) && !defined($trilinears[$i]) or die;
                $i == 590 or die;
                $trilinears[$i] = [javafy($1,$i)];
                if ($noBarycentrics)
                {
                    $noBarycentrics = 0;
                    $i = undef;
                }
            }
            elsif ($line =~ /^Barycentrics (.*)\s*$/
                && $line !~ /^Barycentrics for /) # XXX fooey, X(402)
            {
                defined($i) && defined($trilinears[$i])
                            && !defined($barycentrics[$i]) or die;
                $barycentrics[$i] = [javafy($1,$i)];

                $i = undef;
            }
            elsif ($line =~ /^<h3/)
            {
                die;
            }
        }
    }
    my $n = @names-1;

    #
    # For those with auxiliary functions (e.g. "where f = ..."),
    # break them out.
    #
    for my $i (1..$n)
    {
        if ($trilinears[$i][0] =~ /where/)
        {
            # XXX implement me
        }
        if (defined($barycentrics[$i][0]))
        {
            if ($barycentrics[$i][0] =~ /where/)
            {
                # XXX implement me
            }
        }
        else
        {
            $barycentrics[$i] = []; # so it won't be undefined, just 0 length
        }
    }

    sub splitDefinitions($)
    {
        my ($def) = @_;
        $def =~ s/( where [^=]*)=/$1EQUALS/g; # hide ='s after wheres, so we don't split on them

        if (1) # XXX can't differentiate between types of ='s after wheres, e.g. x(129), so bail for now
        {
            while ($def =~ s/( where [^=]*)=/$1EQUALS/g)
            {
                # nothing
            }
        }


        my @list = split(/=/, $def);
        for my $j (0..@list-1)
        {
            $list[$j] =~ s/EQUALS/=/g;
        }
        return @list;
    }

    #
    # For those with multiple supposedly equivalent definitions,
    # break them up
    #
    for my $i (1..$n)
    {
        if (!defined($names[$i]))
        {
            die "names[$i] was not defined!?";
        }
        if (!defined($trilinears[$i][0]))
        {
            die "trilinears[$i][0] was not defined!?";
        }
        $trilinears[$i] = [splitDefinitions($trilinears[$i][0])];
        #$barycentrics[$i] = [splitDefinitions($barycentrics[$i][0])];
    }





    if ($verbose >= 1)
    {
        print "/*\n";
        for my $i (1..$n)
        {
            print "$i:\n";
            print "\tname='$names[$i]'\n";
            for my $j (0..@{$trilinears[$i]}-1)
            {
                print "\tt='$trilinears[$i][$j]'\n";
            }
            for my $j (0..@{$barycentrics[$i]}-1)
            {
                print "\tb='$barycentrics[$i][$j]'\n";
            }
        }
        print "*/\n\n";
    }




print "// AUTOMATICALLY GENERATED-- DO NOT EDIT!\n" x 40;
print <<END;

#include "macros.h" // for assert

public class ETCcontents
    implements ETC.Privates
{
    private static final String names[] = {
        null,
END
        for my $i (1..$n)
        {
            print "        /* $i */ \"$names[$i]\",\n";
        }
print <<END;
    }; // names
    private static final int numTrilinears[] = {
        -1,
END
        for my $i (1..$n)
        {
            print "        /* $i */ ".(0+@{$trilinears[$i]}).",\n";
        }
print <<END;
    }; // numTrilinears
    private static final int numBarycentrics[] = {
        -1,
END
        for my $i (1..$n)
        {
            print "        /* $i */ ".(0+@{$barycentrics[$i]}).",\n";
        }
print <<END;
    }; // numBarycentrics

END

# Break up into chunks,
# to avoid jikes error "*** Semantic Error: Method "double[] calcTrilinears0(int i, int j, double a, double b, double c);" in type "ETCcontents" produced a code attribute that exceeds the code limit of 65535 elements.
# (it does that if one too many cases; if more than that, it goes
# into an endless loop!)
# max usable chunkSize seems to be 1199 as of this writing,
# but it may become less as I implement more of the cases.

my $chunkSize = 1000;
my $nChunks = int(($n-1)/$chunkSize)+1;

print <<END;
    public double[] calcTrilinears(int i, int j, double a, double b, double c)
    {
        assert(INRANGE(1 <=, i, <= getNumCenters()));
        assert(INRANGE(0 <=, j, < numTrilinears[i]));
        switch ((i-1)/$chunkSize)
        {
END
            for my $iChunk (0..$nChunks-1)
            {
                print("            case $iChunk: return calcTrilinears$iChunk(i, j, a, b, c);\n");
            }
print <<END;
        }
        assert(false); // can't get here
        return null; // shut up compiler
    } // calcTrilinears

    public double[] calcBarycentrics(int i, int j, double a, double b, double c)
    {
        assert(INRANGE(1 <=, i, <= getNumCenters()));
        assert(INRANGE(0 <=, j, < numBarycentrics[i]));
        switch ((i-1)/$chunkSize)
        {
END
            for my $iChunk (0..$nChunks-1)
            {
                print("            case $iChunk: return calcBarycentrics$iChunk(i, j, a, b, c);\n");
            }
print <<END;
        }
        assert(false); // can't get here
        return null; // shut up compiler
    } // calcBarycentrics
END

  for my $iChunk (0..$nChunks-1)
  {
    my $iMin = $iChunk*$chunkSize + 1;
    my $iMax = $iChunk*$chunkSize + $chunkSize;
    if ($iMax > $n)
    {
        $iMax = $n;
    }

print<<END;
    private double[] calcTrilinears$iChunk(int i, int j, double a, double b, double c)
    {
        assert(INRANGE(1 <=, i, <= getNumCenters()));
        assert(INRANGE(0 <=, j, < numTrilinears[i]));

        // angles...
        double A = Math.acos((b*b + c*c - a*a) / (2*b*c));
        double B = Math.acos((c*c + a*a - b*b) / (2*c*a));
        double C = Math.acos((a*a + b*b - c*c) / (2*a*b));

        double area = area(a,b,c);

        double W = (a*a + b*b + c*c) / (4*area);
        double omega = Math.atan(1./W); // Brocard angle

        // described under X(370)
        double S_A = (b*b + c*c - a*a)/2;
        double S_B = (c*c + a*a - b*b)/2;
        double S_C = (a*a + b*b - c*c)/2;

        switch (i)
        {
END
    for my $i ($iMin..$iMax)
    {
        print "            case $i: // \"$names[$i]\"\n";
        print "            {\n";
        print "                switch (j)\n";
        print "                {\n";
        for my $j (0..@{$trilinears[$i]}-1)
        {
            print "                    case $j:\n";

            if ($trilinears[$i][$j] eq ""
             || $trilinears[$i][$j] =~ /where/ # can't handle yet
             || $trilinears[$i][$j] =~ /sought/
             || $trilinears[$i][$j] =~ /see below/
             || $trilinears[$i][$j] =~ /see note above/

             || $trilinears[$i][$j] =~ /\ba?x\b/ # can't handle yet
             || $trilinears[$i][$j] =~ /\bu\b/ # can't handle yet
             #|| $j > 0 # XXX get rid
             || 0)
            {
                print "                            // \"$trilinears[$i][$j]\"\n";
                print "                            return null;\n";
            }
            else
            {
                print "                            return new double[] {$trilinears[$i][$j]};\n";
            }
        }
        print "                }\n";
        print "                break;\n";
        print "            }\n";
    }
    print <<END;
        } // switch (i)
        assert(false); // can't get here
        return null; // shut up compiler
    } // calcTrilinears$iChunk

    private double[] calcBarycentrics$iChunk(int i, int j, double a, double b, double c)
    {
        assert(INRANGE(1 <=, i, <= getNumCenters()));
        assert(INRANGE(0 <=, j, < getNumBarycentrics(i)));

        // angles...
        double A = Math.acos((b*b + c*c - a*a) / (2*b*c));
        double B = Math.acos((c*c + a*a - b*b) / (2*c*a));
        double C = Math.acos((a*a + b*b - c*c) / (2*a*b));

        double area = area(a,b,c);

        double W = (a*a + b*b + c*c) / (4*area);
        double omega = Math.atan(1./W); // Brocard angle

        // described under X(370)
        double S_A = (b*b + c*c - a*a)/2;
        double S_B = (c*c + a*a - b*b)/2;
        double S_C = (a*a + b*b - c*c)/2;

        switch (i)
        {
END
    for my $i ($iMin..$iMax)
    {
        print "            case $i: // \"$names[$i]\"\n";
        print "            {\n";
        print "                switch (j)\n";
        print "                {\n";
        for my $j (0..@{$barycentrics[$i]}-1)
        {
            print "                    case $j:\n";

            if ($barycentrics[$i][$j] eq ""
             || $barycentrics[$i][$j] =~ /where/ # can't handle yet
             || $barycentrics[$i][$j] =~ /sought/
             || $barycentrics[$i][$j] =~ /see below/
             || $barycentrics[$i][$j] =~ /see note above/

             || $barycentrics[$i][$j] =~ /\b[fguxL]\b/ # defined in a "where" in trilinear definition above; can't handle yet
             || $j > 0 # XXX get rid
             || 0)
            {
                print "                            // \"$barycentrics[$i][$j]\"\n";
                print "                            return null;\n";
            }
            else
            {
                print "                            return new double[] {$barycentrics[$i][$j]};\n";
            }
        }
        print "                }\n";
        print "                break;\n";
        print "            }\n";
    }
    print <<END;
        } // switch (i)
        assert(false); // can't get here
        return null; // shut up compiler
    } // calcBarycentrics$iChunk
END
  } # for iChunk

print <<END;

    private static double sin2(double x) { double temp = Math.sin(x); return temp*temp; }
    private static double cos2(double x) { double temp = Math.cos(x); return temp*temp; }
    private static double tan2(double x) { double temp = Math.tan(x); return temp*temp; }
    private static double cot2(double x) { double temp = cot(x); return temp*temp; }
    private static double sec2(double x) { double temp = sec(x); return temp*temp; }
    private static double csc2(double x) { double temp = csc(x); return temp*temp; }
    private static double sin3(double x) { double temp = Math.sin(x); return temp*temp*temp; }
    private static double cos3(double x) { double temp = Math.cos(x); return temp*temp*temp; }
    private static double tan3(double x) { double temp = Math.tan(x); return temp*temp*temp; }
    private static double sec3(double x) { double temp = sec(x); return temp*temp*temp; }
    private static double csc3(double x) { double temp = csc(x); return temp*temp*temp; }
    private static double sec4(double x) { double temp = sec(x); return temp*temp*temp*temp; }
    private static double csc4(double x) { double temp = csc(x); return temp*temp*temp*temp; }
    private static double cot(double x) { return 1./Math.tan(x); }
    private static double sec(double x) { return 1./Math.cos(x); }
    private static double csc(double x) { return 1./Math.sin(x); }

    private static double area(double a, double b, double c)
    {
        double s = (a+b+c)/2;
        return Math.sqrt(s*(s-a)*(s-b)*(s-c)); // Heron's formula
    }




    // XXX whatever this means...
    public boolean isOnLineAtInfinity(int i)
    {
        return false
            || i==30
            || (i >= 511 && i <= 526)
            || (i >= 527 && i <= 545)
            || (i >= 674 && i <= 674)
            || i == 680
            || i == 688
            || i == 690
            || (i >= 696 && i <= 726) && i%2==0
            || (i >= 730 && i <= 736) && i%2==0
            || (i >= 740 && i <= 746) && i%2==0
            || (i >= 752 && i <= 754) && i%2==0
            || (i >= 758 && i <= 760) && i%2==0
            || (i >= 766 && i <= 768) && i%2==0
            || i == 772
            || (i >= 776 && i <= 796) && i%2==0
            || (i >= 802 && i <= 808) && i%2==0
            || (i >= 812 && i <= 818) && i%2==0
            || (i >= 824 && i <= 826) && i%2==0
            || (i >= 830 && i <= 834) && i%2==0
            || i == 838
            || i == 888
            || i == 891
            || i == 900
            || i == 912
            || i == 916
            || i == 918
            || i == 924
            || i == 926
            || i == 928
            || i == 952
            || i == 971
            ;
    } // isOnLineAtInfinity

    public int getNumCenters()
    {
        return $n;
    }
    public String getName(int i)
    {
        return names[i];
    }
    public int getNumTrilinears(int i)
    {
        return numTrilinears[i];
    }
    public int getNumBarycentrics(int i)
    {
        return numBarycentrics[i];
    }
    public int getNumTrilinearsOrBarycentrics(int i, int j)
    {
        return j == 0 ? getNumTrilinears(i)
                      : getNumBarycentrics(i);
    }
    public double[] calcCenter(int i, int j, int k,
                                     double A[], double B[], double C[])
    {
        double a = VecMath.dist(B, C);
        double b = VecMath.dist(C, A);
        double c = VecMath.dist(A, B);
        double barycentrics[];
        if (j == 0) // trilinear
        {
            double trilinears[] = calcTrilinears(i, k, a, b, c);
            barycentrics = new double[]{a*trilinears[0],
                                        b*trilinears[1],
                                        c*trilinears[2]};
        }
        else // barycentric
        {
            barycentrics = calcBarycentrics(i, k, a, b, c);
        }
        double result[] = VecMath.vxm(barycentrics, new double[][]{A,B,C});
        VecMath.vxs(result, result, 1./(barycentrics[0]+barycentrics[1]+barycentrics[2]));
        return result;
    } // calcCenter

    public void increment(int ind[/*3*/])
    {
        do
        {
            ++ind[2];
            while (ind[2] == getNumTrilinearsOrBarycentrics(ind[0],ind[1]))
            {
                ++ind[1];
                ind[2] = 0;
                while (ind[1] == 2)
                {
                    ind[1] = 0;
                    ++ind[0];
                    if (ind[0] == getNumCenters()+1)
                    {
                        ind[0] = 1; // 1-based indexing
                    }
                }
            }
            assert(INRANGE(1 <=, ind[0], <= getNumCenters()));
            assert(INRANGE(0 <=, ind[1], <= 1));
            assert(INRANGE(0 <=, ind[2], < getNumTrilinearsOrBarycentrics(ind[0],ind[1])));
        }
        while ((ind[1] == 0 ? calcTrilinears(ind[0], ind[2], 1.,1.,1.)
                            : calcBarycentrics(ind[0], ind[2], 1.,1.,1.)) == null);
    } // increment
    public void decrement(int ind[/*3*/])
    {
        do
        {
            --ind[2];
            while (ind[2] == -1)
            {
                --ind[1];
                while (ind[1] == -1)
                {
                    --ind[0];
                    ind[1] = 2-1;
                    if (ind[0] == 0) // skip 0
                    {
                        ind[0] = getNumCenters(); // 1-based indexing
                    }
                }
                ind[2] = getNumTrilinearsOrBarycentrics(ind[0],ind[1])-1;
            }
            assert(INRANGE(1 <=, ind[0], <= getNumCenters()));
            assert(INRANGE(0 <=, ind[1], <= 1));
            assert(INRANGE(0 <=, ind[2], < getNumTrilinearsOrBarycentrics(ind[0],ind[1])));
        }
        while ((ind[1] == 0 ? calcTrilinears(ind[0], ind[2], 1.,1.,1.)
                            : calcBarycentrics(ind[0], ind[2], 1.,1.,1.)) == null);
    } // decrement



    //
    // Compute and print the searchable index of the ETC...
    //
    public void main(String args[])
    {
        int nCenters = getNumCenters();
        double kxs[][/*3*/] = new double[nCenters][3];

        {
            int maxNumTrilinears = 0;
            int maxNumBarycentrics = 0;
            int i;
            for (i = 1; i <= nCenters; ++i) // sic
            {
                maxNumTrilinears = Math.max(maxNumTrilinears, numTrilinears[i]);
                maxNumBarycentrics = Math.max(maxNumBarycentrics, numBarycentrics[i]);
            }
            PRINT(maxNumTrilinears);
            PRINT(maxNumBarycentrics);
        }


        double a = 6, b = 9, c = 13;
        int i;
        for (i = 1; i <= nCenters; ++i) // sic
        {

            double T[] = calcTrilinears(i, 0, a, b, c);
            double B[] = calcBarycentrics(i, 0, a, b, c);
            if (T != null && T.length != 3)
                System.err.println("WARNING: trilinears "+i+" have length "+T.length);
            if (B != null && B.length != 3)
                System.err.println("WARNING: barycentrics "+i+" have length "+B.length);
            if (B != null && T == null)
            {
                //System.err.println("NOTICE: barycentrics "+i+" exists but triliners doesn't");
            }
            if (T != null && B == null)
            {
                //System.err.println("NOTICE: trilinears "+i+" exists but barycentrics doesn't");
            }
            System.out.println(i+":" + (isOnLineAtInfinity(i) ? "(infinite)" : ""));
            double area = area(a,b,c);
            double kx_from_T = Double.NaN, kx_from_B = Double.NaN;
            if (T != null)
            {
                double x = T[0], y = T[1], z = T[2];
                double k = (isOnLineAtInfinity(i) ? 1/x+1/y+1/z
                                                  : 2*area/(a*x + b*y + c*z));
                kx_from_T = k * x;
                System.out.println("    from trilinears:  " + kx_from_T);
                assert(!Double.isNaN(kx_from_T));
                if (kx_from_T <= -469
                 || kx_from_T >= 465)
                {
                    System.err.println("WARNING: "+i+(isOnLineAtInfinity(i)? " (infinite)" : "")+": out of range");
                    System.err.println("    from trilinears:  " + kx_from_T);
                }
            }
            if (B != null)
            {
                double x = B[0]/a;
                double k = (isOnLineAtInfinity(i) ? a/B[0]+b/B[1]+c/B[2]
                                                  : 2*area/(B[0] + B[1] + B[2]));
                kx_from_B = k * x;
                System.out.println("    from barycentric: " + kx_from_B);
                assert(!Double.isNaN(kx_from_B));
                if (kx_from_B <= -469
                 || kx_from_B >= 465)
                {
                    System.err.println("WARNING: "+i+(isOnLineAtInfinity(i)? " (infinite)" : "")+": out of range");
                    System.err.println("    from barycentrics:  " + kx_from_B);
                }
            }
            if (T != null
             && B != null
             && kx_from_T != kx_from_B // so -Infinity == -Infinity is okay
             && !(Math.abs(kx_from_T - kx_from_B) < 1e-10))
            {
                System.err.println("WARNING: "+i+(isOnLineAtInfinity(i)? " (infinite)" : "")+": two different answers");
                System.err.println("    from trilinears:  " + kx_from_T);
                System.err.println("    from barycentric: " + kx_from_B);
            }
            kxs[i-1][0] = kx_from_T;
            kxs[i-1][1] = kx_from_B;
            kxs[i-1][2] = (double)i;
        }
        SortStuff.sort(kxs, new SortStuff.Comparator() {
            public int compare(Object a, Object b)
            {
                double xxx[/*3*/] = (double [])a;
                double yyy[/*3*/] = (double [])b;

                double x = (!Double.isNaN(xxx[0]) ? xxx[0] : xxx[1]);
                double y = (!Double.isNaN(yyy[0]) ? yyy[0] : yyy[1]);

                // NaNs go at the end...
                if (!Double.isNaN(x) && Double.isNaN(y))
                    return -1; // in order
                if (Double.isNaN(x) && !Double.isNaN(y))
                    return 1; // out of order
                if (!Double.isNaN(x) && !Double.isNaN(y))
                {
                    if (x < y)
                        return -1; // in order
                    if (x > y)
                        return 1; // out of order
                }

                // if otherwise equal, base on index
                if (xxx[2] < yyy[2])
                    return -1; // in order
                if (xxx[2] > yyy[2])
                    return 1; // out of order;
                /*
                System.err.println("comparing "+a+" with "+b+"!?");
                System.err.println("comparing "+xxx+" with "+yyy+"!?");
                System.err.println("comparing "+x+" with "+y+"!?");
                System.err.println("comparing "+xxx[2]+" with "+yyy[2]+"!?");
                assert(false); // should never return 0
                */
                return 0;
            }
        });
        System.out.println("coordinate from trilinear"
                         + "  "
                         + "coordinate from barycentric"
                         + "  "
                         + "index"
                         + "  "
                         + "rank");
        FOR (i, nCenters)
        {
            System.out.println(kxs[i][0]
                             + "    "
                             +(Math.abs(kxs[i][1]-kxs[i][0]) < 1e-10 ? "==" : "!=")
                             + "    "
                             + kxs[i][1]
                             + "    "
                             + (int)kxs[i][2]
                             +"    "
                             + (i+1));
        }
        // XXX the above is convoluted... make a little class instead, I think
    } // main

} // class ETCcontents
END


}
